<?php
/**
 * NHT Trading Academy GetLinked Support Plugin
 *
 * @package NHT_Trading_Academy_GetLinked_Support_Plugin
 * @author New Horizon Technology Ltd.
 * @copyright 2024 New Horizon Technology Ltd.
 * @license All rights reserved.
 */

defined( 'ABSPATH' ) or die( 'Access Denied' );

/**
 * The function which adds rewrite rules for handling the automatic login & redirection.
 */
function nht_trading_academy_get_linked_support_plugin_add_rewrite_rules() {
	add_rewrite_rule( 'from-landing[/]?$', 'index.php?from_landing=1', 'top' );
	if ( ! get_option( 'nht_trading_academy_get_linked_support_permalinks_flushed' ) ) {

		flush_rewrite_rules( false );
		update_option( 'nht_trading_academy_get_linked_support_permalinks_flushed', 1 );
	}
}

add_action( 'init', 'nht_trading_academy_get_linked_support_plugin_add_rewrite_rules' );

/**
 * The function which adds the query param to the list of query vars.
 *
 * @param array $vars - the current list of query vars.
 */
function nht_trading_academy_get_linked_support_plugin_add_query_vars( $vars ) {
	$vars[] = 'from_landing';
	return $vars;
}

add_filter( 'query_vars', 'nht_trading_academy_get_linked_support_plugin_add_query_vars' );

/**
 * The function which handles the automatic login & redirection.
 * Hooks to the `pre_get_posts` action.
 *
 * @param WP_Query $query - the current query object.
 */
function nht_trading_academy_get_linked_support_plugin_check_and_handle_auto_login( $query ) {
	if ( ! is_admin() && $query->is_main_query() && $query->get( 'from_landing' ) ) {
		$options = get_options(
			array(
				'nht_trading_academy_redirectable_post_types',
				'nht_trading_academy_default_redirection_post',
				'nht_trading_academy_default_products',
				'nht_trading_academy_advertiser_hashes',
			)
		);
		if ( ! is_array( $options['nht_trading_academy_redirectable_post_types'] ) ) {
			$options['nht_trading_academy_redirectable_post_types'] = array( 'post', 'page' );
		}
		if ( false === $options['nht_trading_academy_default_redirection_post'] ) {
			$options['nht_trading_academy_default_redirection_post'] = get_option( 'page_on_front' );
		} else {
			$options['nht_trading_academy_default_redirection_post'] = (int) $options['nht_trading_academy_default_redirection_post'];
		}
		if ( ! is_array( $options['nht_trading_academy_default_products'] ) ) {
			$options['nht_trading_academy_default_products'] = array();
		} else {
			$options['nht_trading_academy_default_products'] = array_map( 'intval', $options['nht_trading_academy_default_products'] );
		}
		if ( ! is_array( $options['nht_trading_academy_advertiser_hashes'] ) ) {
			$options['nht_trading_academy_advertiser_hashes'] = array();
		}
		$opts = array(
			'to'             => $options['nht_trading_academy_default_redirection_post'],
			'products'       => $options['nht_trading_academy_default_products'],
			'email'          => array_key_exists( 'email', $_GET ) ? $_GET['email'] : '',
			'advertiserhash' => array_key_exists( 'advertiserhash', $_GET ) ? $_GET['advertiserhash'] : '',
		);
		if ( array_key_exists( 'to', $_GET ) ) {
			$to_post = get_post( (int) $_GET['to'] );
			if ( $to_post && in_array( $to_post->post_type, $options['nht_trading_academy_redirectable_post_types'], true ) ) {
				$opts['to'] = $to_post->ID;
			}
		}
		$valid = true;
		if ( empty( $opts['email'] ) || ! is_email( $opts['email'] ) ) {
			$valid = false;
		}
		if ( empty( $opts['advertiserhash'] ) || ! in_array( $opts['advertiserhash'], $options['nht_trading_academy_advertiser_hashes'], true ) ) {
			$valid = false;
		}
		$post_to_redirect_to = get_post( $opts['to'] );
		if ( ! $post_to_redirect_to || ! in_array( $post_to_redirect_to->post_type, $options['nht_trading_academy_redirectable_post_types'], true ) ) {
			$post_to_redirect_to = null;
		}
		if ( $valid ) {
			$user = get_user_by( 'email', $opts['email'] );
			if ( $user ) {
				wp_set_current_user( $user->ID, $user->user_login );
				wp_set_auth_cookie( $user->ID );
				do_action( 'wp_login', $user->user_login, $user );
			}
			if ( function_exists( 'WC' ) && ! empty( $opts['products'] ) ) {
				$cart = WC()->cart;
				$cart->empty_cart();
				foreach ( $opts['products'] as $product_id ) {
					$cart->add_to_cart( $product_id );
				}
			}
		}
		$final_redirection_url = null === $post_to_redirect_to ? get_home_url() : get_permalink( $post_to_redirect_to );
		wp_redirect( $final_redirection_url );
		exit;
	}
}

add_action( 'pre_get_posts', 'nht_trading_academy_get_linked_support_plugin_check_and_handle_auto_login' );
